<?php
require __DIR__ . '/AllohaClient.php';

function loadAllohaConfig(): array {
  // Procura o config.php subindo diretórios (funciona em /public_html/giga/alloha-vendas)
  $candidates = [
    __DIR__ . '/../../alloha_private/config.php',           // /public_html/giga/alloha_private (não recomendado)
    __DIR__ . '/../../../alloha_private/config.php',        // /public_html/alloha_private (não recomendado)
    __DIR__ . '/../../../../alloha_private/config.php',     // /home/USER/alloha_private (recomendado)
    dirname(__DIR__, 2) . '/alloha_private/config.php',
    dirname(__DIR__, 3) . '/alloha_private/config.php',
    dirname(__DIR__, 4) . '/alloha_private/config.php',
  ];
  foreach ($candidates as $p) {
    if (file_exists($p)) return require $p;
  }
  throw new Exception('Config não encontrado. Coloque em /home/SEUUSUARIO/alloha_private/config.php');
}

$cfg = loadAllohaConfig();

// Ajuste automático para /home/USER/alloha_private/config.php
$api = new AllohaClient($cfg);

function digits(string $v): string { return preg_replace('/\D+/', '', $v); }


function callWithPrefixes(array $cfg, callable $callable): array {
  $prefixes = [
    $cfg['api_prefix'] ?? '/partner',
    '/sale-partner',
    '/sale-partner-api',
    '/sale-partner/partner',
    '/sale-partner-api/partner'
  ];
  $lastEx = null;

  foreach ($prefixes as $pref) {
    $cfg2 = $cfg;
    $cfg2['api_prefix'] = $pref;
    $api2 = new AllohaClient($cfg2);
    try {
      $res = $callable($api2);
      return ['result' => $res, 'cfg' => $cfg2];
    } catch (Exception $e) {
      $lastEx = $e;
      if (stripos($e->getMessage(), 'API HTTP 404') === false) {
        throw $e;
      }
    }
  }

  throw $lastEx ?: new Exception('Falha ao chamar API com prefixos alternativos.');
}


function friendlyError(string $msg): string {
  // Mensagens amigáveis sem vazar detalhes sensíveis
  if (stripos($msg, '401') !== false || stripos($msg, '403') !== false) return 'Falha de autenticação na API. Verifique credenciais/ambiente.';
  if (stripos($msg, 'timeout') !== false) return 'A API demorou para responder. Tente novamente.';
  if (stripos($msg, 'lead') !== false) return 'Endereço sem viabilidade para reserva (lead).';
  return 'Não foi possível concluir agora. Confira os dados e tente novamente.';
}

function page(string $title, string $body): void {
  $css = file_get_contents(__DIR__ . '/style.css');
  header('Content-Type: text/html; charset=utf-8');
  echo "<!doctype html><html lang='pt-br'><head><meta charset='utf-8'><meta name='viewport' content='width=device-width,initial-scale=1'><title>{$title}</title><style>{$css}</style></head><body>{$body}</body></html>";
  exit;
}

try {
  // Campos
  $userType = $_POST['userType'] ?? 'NATURAL_PERSON';
    $productId = trim($_POST['productId'] ?? '');
  $planUpgradeOptionId = trim($_POST['planUpgradeOptionId'] ?? '');

  $zipCode = digits($_POST['zipCode'] ?? '');
  $street = trim($_POST['street'] ?? '');
  $number = trim($_POST['number'] ?? '');
  $neighborhood = trim($_POST['neighborhood'] ?? '');
  $city = trim($_POST['city'] ?? '');
  $state = strtoupper(trim($_POST['state'] ?? ''));
  $complement = trim($_POST['complement'] ?? '');
  $reference = trim($_POST['reference'] ?? '');

  $fullName = trim($_POST['fullName'] ?? '');
  $document = digits($_POST['document'] ?? '');
  $birthDate = trim($_POST['birthDate'] ?? '');
  $phone = digits($_POST['phone'] ?? '');
  $email = trim($_POST['email'] ?? '');

  if ($productId === '') throw new Exception('Campo obrigatório: productId.');
  if ($zipCode === '' || $street === '' || $number === '' || $city === '' || $state === '') throw new Exception('Endereço incompleto.');
  if ($fullName === '' || $document === '' || $phone === '' || $email === '') throw new Exception('Dados do cliente incompletos.');

  $token = $api->getAccessToken();

  // 0) Consulta de viabilidade para obter locationId automaticamente
  $tmp = callWithPrefixes($cfg, function($apiTmp) use ($token, $zipCode, $street, $number, $neighborhood, $city, $state) {
    return $apiTmp->checkAvailability($token, [
    'zipCode' => $zipCode,
    'street'  => $street,
    'number'  => $number,
    'neighborhood' => $neighborhood,
    'city'    => $city,
    'state'   => $state,
    ]);
  });
  $avail = $tmp['result'];
  $cfg = $tmp['cfg'];
  $api = new AllohaClient($cfg);

  // Tenta extrair locationId de diferentes formatos de resposta
  $locationId = $avail['locationId'] ?? null;

  if ($locationId === null && isset($avail[0]) && is_array($avail[0])) {
    $locationId = $avail[0]['locationId'] ?? null;
  }
  if ($locationId === null && isset($avail['data'])) {
    if (is_array($avail['data'])) {
      $locationId = $avail['data']['locationId'] ?? null;
      if ($locationId === null && isset($avail['data'][0]) && is_array($avail['data'][0])) {
        $locationId = $avail['data'][0]['locationId'] ?? null;
      }
    }
  }

  // Checagens básicas de disponibilidade
  $availStatus = $avail['status'] ?? ($avail['data']['status'] ?? null);
  $isAvailable = $avail['available'] ?? ($avail['isAvailable'] ?? null);

  if ($locationId === null) {
    throw new Exception('Não foi possível obter locationId na consulta de disponibilidade.');
  }
  // Se a API informar indisponível, já interrompe
  if ($isAvailable === false) {
    throw new Exception('lead');
  }
  if (is_string($availStatus) && stripos($availStatus, 'UNAVAILABLE') !== false) {
    throw new Exception('lead');
  }

  // 1) Register

  $registerPayload = [
    'transactionId' => null,
    'locationId' => (string)$locationId,
    'productId'  => (int)$productId,
    'userType'   => $userType
  ];
  if ($planUpgradeOptionId !== '') $registerPayload['planUpgradeOptionId'] = (int)$planUpgradeOptionId;

  $register = $api->registerCheckout($token, $registerPayload);
  $transactionId = $register['transactionId'] ?? ($register['id'] ?? null);
  if (!$transactionId) throw new Exception('transactionId não retornado no register.');

  // 2) Address
  $addr = $api->sendAddress($token, $transactionId, [
    'zipCode' => $zipCode,
    'street'  => $street,
    'number'  => $number,
    'neighborhood' => $neighborhood,
    'city'    => $city,
    'state'   => $state,
    'complement' => $complement,
    'reference' => $reference,
  ]);

  if (!empty($addr['lead'])) {
    throw new Exception('lead');
  }

  // 3) Personal Data (mínimo). Ajuste se sua operação exigir campos extras.
  $personal = [
    'fullName' => $fullName,
    'document' => $document,
    'phone'    => $phone,
    'email'    => $email,
  ];
  if ($birthDate !== '') $personal['birthDate'] = $birthDate;

  $api->sendPersonalData($token, $transactionId, $personal);

  // 4) Status imediato
  $status = $api->analysisStatus($token, $transactionId);
  $decision = $status['decision'] ?? 'UNKNOWN';

  $body = "<h2>✅ Venda iniciada</h2>";
  $body .= "<div class='card ok'><b>Transaction ID:</b> " . htmlspecialchars($transactionId) . "<br><b>Status análise:</b> " . htmlspecialchars($decision) . "</div>";
  $body .= "<div class='card'><a href='status.php?apiPrefix=" . urlencode($cfg['api_prefix'] ?? '') . "&transactionId=" . urlencode($transactionId) . "'>👉 Abrir tela de status</a></div>";
  $body .= "<div class='card'><a href='finalizar.php?apiPrefix=" . urlencode($cfg['api_prefix'] ?? '') . "&transactionId=" . urlencode($transactionId) . "'>👉 Ir para finalizar (agenda)</a></div>";
  $body .= "<p class='small'><a href='index.php'>Voltar</a></p>";

  page('Venda iniciada', $body);

} catch (Exception $e) {
  $api->logApp($e->getMessage());

  $code = substr(md5($e->getMessage()), 0, 8);
  $msg = friendlyError($e->getMessage());

  $body = "<h2>❌ Erro</h2>";
  $body .= "<div class='card bad'><b>{$msg}</b><div class='small'>Código: {$code}</div></div>";
  $body .= "<p class='small'><a href='index.php'>Voltar</a></p>";
  page('Erro', $body);
}
