<?php
require __DIR__ . '/AllohaClient.php';

function loadAllohaConfig(): array {
  // Procura o config.php subindo diretórios (funciona em /public_html/giga/alloha-vendas)
  $candidates = [
    __DIR__ . '/../../alloha_private/config.php',           // /public_html/giga/alloha_private (não recomendado)
    __DIR__ . '/../../../alloha_private/config.php',        // /public_html/alloha_private (não recomendado)
    __DIR__ . '/../../../../alloha_private/config.php',     // /home/USER/alloha_private (recomendado)
    dirname(__DIR__, 2) . '/alloha_private/config.php',
    dirname(__DIR__, 3) . '/alloha_private/config.php',
    dirname(__DIR__, 4) . '/alloha_private/config.php',
  ];
  foreach ($candidates as $p) {
    if (file_exists($p)) return require $p;
  }
  throw new Exception('Config não encontrado. Coloque em /home/SEUUSUARIO/alloha_private/config.php');
}

$cfg = loadAllohaConfig();
$apiPrefix = trim($_GET['apiPrefix'] ?? ($_POST['apiPrefix'] ?? ''));
if ($apiPrefix !== '') { $cfg['api_prefix'] = $apiPrefix; }
$api = new AllohaClient($cfg);
$css = file_get_contents(__DIR__ . '/style.css');

function page(string $title, string $body, string $css): void {
  header('Content-Type: text/html; charset=utf-8');
  echo "<!doctype html><html lang='pt-br'><head><meta charset='utf-8'><meta name='viewport' content='width=device-width,initial-scale=1'><title>{$title}</title><style>{$css}</style></head><body>{$body}</body></html>";
  exit;
}

$transactionId = trim($_GET['transactionId'] ?? ($_POST['transactionId'] ?? ''));

if ($transactionId === '') {
  page('Finalizar', "<h2>Finalizar</h2><div class='card bad'>Informe transactionId.</div><p class='small'><a href='index.php'>Voltar</a></p>", $css);
}

try {
  $token = $api->getAccessToken();

  // Se veio POST, finalizar
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
    $dueDay = (int)($_POST['dueDay'] ?? 0);
    $slot  = $_POST['slot'] ?? ''; // string/obj serializado
    if ($dueDay <= 0) throw new Exception("Selecione um dia de vencimento.");
    if ($slot === '') throw new Exception("Selecione um horário de agendamento.");

    $slotData = json_decode(base64_decode($slot), true);
    if (!is_array($slotData)) throw new Exception("Slot inválido.");

    // Payload do finish - conforme disponibilidade do doc (pode variar).
    // Aqui usamos o que geralmente vem no /bookings.
    $payload = [
      'dueDay' => $dueDay,
      'booking' => $slotData
    ];

    $res = $api->finish($token, $transactionId, $payload);

    $body = "<h2>✅ Finalização enviada</h2>";
    $body .= "<div class='card ok'><b>Transaction ID:</b> " . htmlspecialchars($transactionId) . "</div>";
    $body .= "<div class='card'><b>Resposta:</b><pre style='white-space:pre-wrap'>" . htmlspecialchars(json_encode($res, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)) . "</pre></div>";
    $body .= "<p class='small'><a href='status.php?transactionId=" . urlencode($transactionId) . "'>Voltar para status</a> | <a href='index.php'>Início</a></p>";
    page('Finalizado', $body, $css);
  }

  // GET: buscar bookings e montar form
  $book = $api->getBookings($token, $transactionId);

  $dueDays = $book['dueDays'] ?? ($book['daysDue'] ?? []);
  $slots = $book['availableBookings'] ?? ($book['bookings'] ?? []);

  $body = "<h2>Finalizar (agenda)</h2>";
  $body .= "<div class='card'><b>Transaction ID:</b> " . htmlspecialchars($transactionId) . "</div>";

  if (empty($dueDays) || empty($slots)) {
    $body .= "<div class='card warn'>Ainda não há agenda disponível. Verifique se a análise está aprovada e tente novamente.</div>";
    $body .= "<p class='small'><a href='status.php?transactionId=" . urlencode($transactionId) . "'>Ir para status</a> | <a href='index.php'>Início</a></p>";
    page('Finalizar', $body, $css);
  }

  $body .= "<form method='post' class='card'>";
  $body .= "<input type='hidden' name='transactionId' value='" . htmlspecialchars($transactionId) . "'>";

  $body .= "<label>Dia de vencimento</label><select name='dueDay' required>";
  $body .= "<option value=''>Selecione</option>";
  foreach ($dueDays as $d) {
    $dInt = (int)$d;
    $body .= "<option value='{$dInt}'>{$dInt}</option>";
  }
  $body .= "</select>";

  $body .= "<label>Agendamento</label><select name='slot' required>";
  $body .= "<option value=''>Selecione</option>";
  foreach ($slots as $s) {
    // Mostra um texto amigável, mas envia o JSON inteiro do slot em base64.
    $label = $s['label'] ?? ($s['date'] ?? '') . ' ' . ($s['period'] ?? '') . ' ' . ($s['window'] ?? '');
    $val = base64_encode(json_encode($s, JSON_UNESCAPED_UNICODE));
    $body .= "<option value='" . htmlspecialchars($val, ENT_QUOTES) . "'>" . htmlspecialchars(trim($label)) . "</option>";
  }
  $body .= "</select>";

  $body .= "<button type='submit'>Finalizar venda</button>";
  $body .= "</form>";

  $body .= "<p class='small'><a href='status.php?transactionId=" . urlencode($transactionId) . "'>Voltar para status</a> | <a href='index.php'>Início</a></p>";

  page('Finalizar', $body, $css);

} catch (Exception $e) {
  $api->logApp("finalizar.php: " . $e->getMessage());
  $body = "<h2>❌ Erro</h2><div class='card bad'>" . htmlspecialchars($e->getMessage()) . "</div><p class='small'><a href='index.php'>Voltar</a></p>";
  page('Erro', $body, $css);
}
