<?php
class AllohaClient {
  private array $cfg;

  public function __construct(array $cfg) { $this->cfg = $cfg; }

  private function request(string $method, string $url, ?array $body = null, ?string $token = null): array {
    $ch = curl_init($url);

    $headers = ['Content-Type: application/json'];
    // API KEY (se exigido pelo gateway)
    if (!empty($this->cfg['api_key'])) {
      $headers[] = 'x-api-key: ' . $this->cfg['api_key'];
    }
    if ($token) $headers[] = 'Authorization: Bearer ' . $token;

    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_CUSTOMREQUEST  => $method,
      CURLOPT_HTTPHEADER     => $headers,
      CURLOPT_TIMEOUT        => (int)$this->cfg['timeout'],
    ]);

    if ($body !== null) {
      curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body, JSON_UNESCAPED_UNICODE));
    }

    $raw = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err  = curl_error($ch);
    curl_close($ch);

    if ($raw === false) {
      throw new Exception("Erro CURL: " . $err);
    }

    $json = json_decode($raw, true);
    if (!is_array($json)) {
      // Evita vazar respostas grandes
      $snippet = mb_substr($raw, 0, 600);
      throw new Exception("Resposta inválida (HTTP {$http}): {$snippet}");
    }

    if ($http >= 400) {
      $msg = $json['message'] ?? $json['error'] ?? 'Erro na API';
      throw new Exception("API HTTP {$http} {$method} {$url}: {$msg}");
    }

    return $json;
  }

  private function writeLog(string $file, string $msg): void {
    @file_put_contents($file, '[' . date('c') . '] ' . $msg . PHP_EOL, FILE_APPEND);
  }

  public function logApp(string $msg): void {
    $this->writeLog($this->cfg['log_file'], $msg);
  }

  public function logWebhook(string $msg): void {
    $this->writeLog($this->cfg['webhook_log'], $msg);
  }

  
  private function loadTokenCache(): ?array {
    $path = $this->cfg['token_cache'] ?? null;
    if (!$path || !file_exists($path)) return null;
    $raw = @file_get_contents($path);
    if ($raw === false) return null;
    $data = json_decode($raw, true);
    return is_array($data) ? $data : null;
  }

  private function saveTokenCache(array $data): void {
    $path = $this->cfg['token_cache'] ?? null;
    if (!$path) return;
    // garante diretório
    $dir = dirname($path);
    if (!is_dir($dir)) @mkdir($dir, 0755, true);
    @file_put_contents($path, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
  }

public function getAccessToken(): string {
    $cached = $this->loadTokenCache();
    if ($cached && isset($cached['access_token'], $cached['expires_at']) && time() < (int)$cached['expires_at']) {
      return $cached['access_token'];
    }

    $base = rtrim($this->cfg['auth_base_url'], '/');
    $paths = $this->cfg['auth_login_paths'] ?? ['/authentication/v1/login/sale-partner','/sale-partner-auth/login'];

    $lastEx = null;
    foreach ($paths as $p) {
      $url = $base . '/' . ltrim($p, '/');
      try {
        $res = $this->request('POST', $url, [
          'clientId' => $this->cfg['client_id'],
          'clientSecret' => $this->cfg['client_secret'],
        ], null);

        $token = $res['access_token'] ?? ($res['accessToken'] ?? null);
        $ttl = (int)($res['access_expires_in'] ?? ($res['expires_in'] ?? 300));
        if (!$token) throw new Exception('Resposta de login sem access_token');

        $this->saveTokenCache([
          'access_token' => $token,
          'expires_at' => time() + max(60, $ttl - 30),
        ]);

        return $token;
      } catch (Exception $e) {
        $lastEx = $e;
        if (stripos($e->getMessage(), 'API HTTP 404') === false) {
          throw $e;
        }
      }
    }

    throw $lastEx ?: new Exception('Falha no login');
  }

  private function apiUrl(string $path): string {
    $prefix = $this->cfg['api_prefix'] ?? '/partner';
    $prefix = '/' . trim($prefix, '/');
    return rtrim($this->cfg['api_base_url'], '/') . $prefix . $path;
  }

  // ====== Checkout flow ======
  public function registerCheckout(string $token, array $payload): array {
    return $this->request('POST', $this->apiUrl('/checkouts/register'), $payload, $token);
  }

  public function sendAddress(string $token, string $transactionId, array $payload): array {
    return $this->request('POST', $this->apiUrl("/checkouts/{$transactionId}/address-data"), $payload, $token);
  }

  public function sendPersonalData(string $token, string $transactionId, array $payload): array {
    return $this->request('POST', $this->apiUrl("/checkouts/{$transactionId}/personal-data"), $payload, $token);
  }

  public function analysisStatus(string $token, string $transactionId): array {
    return $this->request('GET', $this->apiUrl("/checkouts/{$transactionId}/analysis-status"), null, $token);
  }


  public function checkAvailability(string $token, array $params): array {
    // GET /partner/v2/availabilities?...
    $qs = http_build_query($params);
    return $this->request('GET', $this->apiUrl('/v2/availabilities') . '?' . $qs, null, $token);
  }

  public function getBookings(string $token, string $transactionId): array {
    return $this->request('GET', $this->apiUrl("/bookings/{$transactionId}"), null, $token);
  }

  public function finish(string $token, string $transactionId, array $payload): array {
    return $this->request('POST', $this->apiUrl("/checkouts/{$transactionId}/finish"), $payload, $token);
  }
}
